from django.shortcuts import render, redirect
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.models import User
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from .models import Note
from django.shortcuts import get_object_or_404


# ---------- REGISTER ----------
def register_view(request):
    if request.method == 'POST':
        username = request.POST.get('username')
        password = request.POST.get('password')

        if User.objects.filter(username=username).exists():
            messages.error(request, "Username already exists")
            return redirect('register')

        user = User.objects.create_user(username=username, password=password)
        messages.success(request, "✅ Account created successfully! Please log in.")
        return redirect('login')

    return render(request, 'register.html')


# ---------- LOGIN ----------
def login_view(request):
    if request.method == 'POST':
        user = authenticate(
            request,
            username=request.POST.get('username'),
            password=request.POST.get('password')
        )
        if user:
            login(request, user)
            messages.success(request, f"👋 Welcome back, {user.username}!")
            return redirect('note_list')
        messages.error(request, "❌ Invalid username or password.")
    return render(request, 'login.html')


# ---------- LOGOUT ----------
def logout_view(request):
    logout(request)
    messages.info(request, "👋 You have been logged out.")
    return redirect('login')


# ---------- LIST NOTES ----------
@login_required
def note_list(request):
    notes = Note.objects.filter(user=request.user)
    return render(request, 'list_note.html', {"notes": notes})


# ---------- ADD NOTE ----------
@login_required
def add_note(request):
    if request.method == 'POST':
        Note.objects.create(
            user=request.user,
            title=request.POST.get('title'),
            content=request.POST.get('content')
        )
        messages.success(request, "📝 Note added successfully!")
        return redirect('note_list')
    return render(request, 'add_note.html')

# ---------- NOTE DETAIL ----------
@login_required
def note_detail(request, slug):
    note = get_object_or_404(Note, slug=slug, user=request.user)
    return render(request, 'note_detail.html', {'note': note})


@login_required
def edit_note(request, slug):
    note = get_object_or_404(Note, slug=slug, user=request.user)
    if request.method == 'POST':
        note.title = request.POST.get('title')
        note.content = request.POST.get('content')
        note.save()
        messages.success(request, "Note updated successfully")
        return redirect('note_list')
    return render(request, 'edit_note.html', {'note': note})

@login_required
def delete_note(request, slug):
    note = get_object_or_404(Note, slug=slug, user=request.user)
    note.delete()
    messages.success(request, "Note deleted successfully")
    return redirect('note_list')
